<?php
/**
 * Plugin Name: WooCommerce Custom Donation
 * Plugin URI: https://www.cybergrapes.com
 * Description: Adds a custom donation product type with user-defined amounts (minimum $5)
 * Version: 1.0.2
 * Author: Cyber Grapes
 * Author URI: https://www.cybergrapes.com
 * Requires at least: 5.8
 * Requires PHP: 7.4
 * WC requires at least: 5.0
 * WC tested up to: 8.0
 * License: GPL v2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain: wc-custom-donation
 */

if (!defined('ABSPATH')) {
    exit;
}

// Check if WooCommerce is active
if (!in_array('woocommerce/woocommerce.php', apply_filters('active_plugins', get_option('active_plugins')))) {
    return;
}

class WC_Custom_Donation {
    
    private $minimum_amount = 5.00;
    
    public function __construct() {
        add_action('init', array($this, 'init'));
        add_filter('product_type_selector', array($this, 'add_donation_product_type'));
        add_action('woocommerce_donation_add_to_cart', array($this, 'donation_add_to_cart_template'));
        add_action('woocommerce_before_add_to_cart_button', array($this, 'display_donation_amount_field'));
        add_action('woocommerce_after_add_to_cart_quantity', array($this, 'display_donation_amount_field_fallback'));
        add_filter('woocommerce_add_to_cart_validation', array($this, 'validate_donation_amount'), 10, 3);
        add_filter('woocommerce_add_cart_item_data', array($this, 'add_donation_amount_to_cart'), 10, 2);
        add_filter('woocommerce_get_cart_item_from_session', array($this, 'get_cart_item_from_session'), 10, 2);
        add_filter('woocommerce_cart_item_price', array($this, 'display_donation_price_in_cart'), 10, 2);
        add_action('woocommerce_before_calculate_totals', array($this, 'update_cart_item_price'));
        add_filter('woocommerce_product_add_to_cart_text', array($this, 'custom_add_to_cart_text'), 10, 2);
        add_filter('woocommerce_is_sold_individually', array($this, 'hide_quantity_field'), 10, 2);
    }
    
    public function init() {
        // Register product type class
        require_once plugin_dir_path(__FILE__) . 'class-wc-product-donation.php';
    }
    
    public function add_donation_product_type($types) {
        $types['donation'] = __('Donation Product', 'wc-custom-donation');
        return $types;
    }
    
    public function donation_add_to_cart_template() {
        wc_get_template('single-product/add-to-cart/simple.php');
    }
    
    public function custom_add_to_cart_text($text, $product) {
        if ($product && $product->get_type() === 'donation') {
            return __('Donate Now', 'wc-custom-donation');
        }
        return $text;
    }
    
    public function display_donation_amount_field() {
        global $product;
        
        if ($product && $product->get_type() === 'donation') {
            echo '<div class="donation-amount-wrapper" style="margin-bottom: 20px; clear: both;">';
            echo '<label for="donation_amount" style="display: block; margin-bottom: 8px; font-weight: bold;">' . __('Enter Donation Amount ($)', 'wc-custom-donation') . '</label>';
            echo '<input type="number" id="donation_amount" name="donation_amount" step="0.01" min="' . esc_attr($this->minimum_amount) . '" value="' . esc_attr($this->minimum_amount) . '" style="width: 100%; max-width: 200px; padding: 10px; font-size: 16px; border: 1px solid #ddd; border-radius: 4px;" required />';
            echo '<p class="donation-minimum-note" style="font-size: 12px; color: #666; margin-top: 5px;">' . sprintf(__('Minimum donation: $%.2f', 'wc-custom-donation'), $this->minimum_amount) . '</p>';
            echo '</div>';
        }
    }
    
    public function display_donation_amount_field_fallback() {
        global $product;
        
        if ($product && $product->get_type() === 'donation') {
            // Only display if not already displayed
            if (!did_action('woocommerce_before_add_to_cart_button')) {
                $this->display_donation_amount_field();
            }
        }
    }
    
    public function hide_quantity_field($return, $product) {
        if ($product && $product->get_type() === 'donation') {
            return true;
        }
        return $return;
    }
    
    public function validate_donation_amount($passed, $product_id, $quantity) {
        $product = wc_get_product($product_id);
        
        if ($product && $product->get_type() === 'donation') {
            if (!isset($_POST['donation_amount']) || empty($_POST['donation_amount'])) {
                wc_add_notice(__('Please enter a donation amount.', 'wc-custom-donation'), 'error');
                return false;
            }
            
            $donation_amount = floatval($_POST['donation_amount']);
            
            if ($donation_amount < $this->minimum_amount) {
                wc_add_notice(sprintf(__('Minimum donation amount is $%.2f', 'wc-custom-donation'), $this->minimum_amount), 'error');
                return false;
            }
        }
        
        return $passed;
    }
    
    public function add_donation_amount_to_cart($cart_item_data, $product_id) {
        $product = wc_get_product($product_id);
        
        if ($product && $product->get_type() === 'donation' && isset($_POST['donation_amount'])) {
            $cart_item_data['donation_amount'] = floatval($_POST['donation_amount']);
            $cart_item_data['unique_key'] = md5(microtime() . rand());
        }
        
        return $cart_item_data;
    }
    
    public function get_cart_item_from_session($cart_item, $values) {
        if (isset($values['donation_amount'])) {
            $cart_item['donation_amount'] = $values['donation_amount'];
        }
        return $cart_item;
    }
    
    public function display_donation_price_in_cart($price, $cart_item) {
        if (isset($cart_item['donation_amount'])) {
            return wc_price($cart_item['donation_amount']);
        }
        return $price;
    }
    
    public function update_cart_item_price($cart) {
        if (is_admin() && !defined('DOING_AJAX')) {
            return;
        }
        
        foreach ($cart->get_cart() as $cart_item_key => $cart_item) {
            if (isset($cart_item['donation_amount'])) {
                $cart_item['data']->set_price($cart_item['donation_amount']);
            }
        }
    }
}

// Initialize the plugin
new WC_Custom_Donation();

// Create the custom product type class file content
// Save this as class-wc-product-donation.php in the same directory
/*
<?php
if (!defined('ABSPATH')) {
    exit;
}

class WC_Product_Donation extends WC_Product {
    
    public function __construct($product) {
        $this->product_type = 'donation';
        $this->supports[] = 'ajax_add_to_cart';
        parent::__construct($product);
    }
    
    public function get_type() {
        return 'donation';
    }
    
    public function is_purchasable() {
        return true;
    }
    
    public function is_sold_individually() {
        return true;
    }
    
    public function is_virtual() {
        return true;
    }
    
    public function needs_shipping() {
        return false;
    }
    
    public function add_to_cart_url() {
        return get_permalink($this->get_id());
    }
    
    public function add_to_cart_text() {
        return __('Donate Now', 'wc-custom-donation');
    }
}
*/